<?php

namespace App\Console\Commands;

use Carbon\Carbon;
use App\Models\Order;
use App\Models\Payment;
use App\Models\Voucher;
use App\Models\Variation;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class CancelledOrderRestockCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:restock-cancelled-orders';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Command for cancel order';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        Log::info("Starting Restock for Orders if payment not completed within 1 minute");

        // Fetch orders with pending payment, not marked as automatic, and created within the last 1 minute
        $thresholdTime = Carbon::now()->subMinutes(1);
        $pendingOrders = Order::where('status', 0) // Status 0 means pending
            ->whereNotNull('voucher_id') // Ensure they have a voucher
            ->whereNull('automatic') // Ensure 'automatic' is null
            ->where('created_at', '<=', $thresholdTime) // Created within the last 1 minute
            ->get();

        if ($pendingOrders->isEmpty()) {
            Log::info("No pending orders found for restocking.");
            return;
        }

        foreach ($pendingOrders as $order) {
            // Calculate the order age in minutes
            $orderAgeInMinutes = Carbon::now()->diffInMinutes($order->created_at);

            // Check if payment is still pending for payment_gateway orders and the order is old enough
            if ($order->payment_type === 'payment_gateway' && $orderAgeInMinutes >= 1) {
                $payment = Payment::where('order_id', $order->id)
                    ->where('payment_status', 'pending')
                    ->first();

                // If payment is still pending after 10 minute, update the order status to canceled (3) and restock the voucher
                if (!$payment || $payment->payment_status === 'pending') {
                    $voucher = Voucher::find($order->voucher_id);
                    if ($voucher) {
                        // Restock the voucher
                        $voucher->status = 0; // Available status
                        $voucher->save();

                        // Increment the stock of the associated variation
                        $variation = Variation::find($voucher->variation_id);
                        if ($variation) {
                            $variation->stock += $order->quantity; // Increment stock based on order quantity
                            $variation->save();
                            Log::info("Restocked Variation ID: {$variation->id} for Voucher ID: {$voucher->id}");
                        } else {
                            Log::warning("Variation not found for Voucher ID: {$voucher->id}");
                        }

                        Log::info("Restocked Voucher ID: {$voucher->id} for Order ID: {$order->id}");
                    } else {
                        Log::warning("Voucher not found for Order ID: {$order->id}");
                    }

                    // Update order status to canceled (status = 3)
                    $order->status = 3;
                    $order->save();

                    Log::info("Order ID: {$order->id} marked as canceled due to payment timeout");
                }
            }
        }

        Log::info("Completed Restock for Orders if payment not completed within 1 minute");
    }
}
