<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Variation;
use Illuminate\Http\Request;


class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->key != null || $request->from != null || $request->to != null) {
            $key = $request->key;
            $from = $request->from;
            $to = $request->to;

            $query = Product::query();
            if (isset($key)) {
                $query->where(function ($q) use ($key) {
                    $q->orWhere('title', 'like', '%'.$key.'%');
                });
            }
            if (isset($from) && isset($to)) {
                $query->where(function ($q) use ($from, $to) {
                    $q->WhereBetween('created_at', [$from, $to]);
                });
            }
            $products = $query->paginate(15);
        } else {
            $products = Product::orderBy('id', 'desc')->paginate(15);
        }

        return view('admin.product.product_list', compact('products'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.product.product_create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'title' => 'required',
            'slug' => 'required',
            'image' => 'required',
            'type' => 'required',
            'status' => 'required',
            'content' => 'required',
        ]);


        $img = time() . '.' . $request->image->getClientOriginalExtension();
        $location = public_path('upload/' . $img);
        $svl = 'public/upload/' . $img;
        request()->image->move(public_path('upload'), $img);

        $product = new Product();
        $product->title = $request->title;
        $product->slug = str_slug($request->slug);
        $product->image = $svl;
        $product->type = $request->type;

        $product->status = $request->status;
        $product->content = $request->content;
        $product->save();

        return redirect('admin/product')->with('success', 'Product added successfully');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $product = Product::find($id);
        $variations = Variation::where('product_id', $id)
        ->orderBy('id', 'desc')
        ->get();

        return view('admin.product.product_edit', compact('product', 'variations'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $this->validate($request, [
            'title' => 'required',
            'slug' => 'required',
            'type' => 'required',
            'status' => 'required',
            'content' => 'required',
        ]);
    
        $product = Product::find($id);
    
        if ($request->hasFile('image')) {
            // Remove old image if it exists
            $oldImagePath = public_path(str_replace('public/', '', $product->image));
            if (file_exists($oldImagePath)) {
                unlink($oldImagePath);
            }
    
            // Save the new image
            $img = time() . '.' . $request->image->getClientOriginalExtension();
            $location = public_path('upload/' . $img);
            $svl = 'public/upload/' . $img;
            $request->image->move(public_path('upload'), $img);
        } else {
            // Keep the old image if no new image is uploaded
            $svl = $product->image;
        }
    
        // Update product details
        $product->title = $request->title;
        $product->slug = str_slug($request->slug);
        $product->image = $svl;
        $product->type = $request->type;
        $product->status = $request->status;
        $product->content = $request->content;
        $product->save();
    
        return redirect('admin/product')->with('success', 'Product updated successfully');
    }
    

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    public function delete($id)
    {
        $product = Product::find($id);
    
        if ($product) {
            // Check if the image file exists and delete it
            $imagePath = public_path(str_replace('public/', '', $product->image));
            if (file_exists($imagePath)) {
                unlink($imagePath);
            }
    
            // Delete the product record from the database
            $product->delete();
            return redirect()->back()->with('success', 'Product deleted successfully');
        }
    
        return redirect()->back()->with('error', 'Product not found');
    }
}
