<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use App\Notifications\RegisterNotification;
use Illuminate\Support\Facades\Notification;

class LoginController extends Controller
{
    public function login()
    {
        return view('auth.login');
    }

    public function register()
    {
        return view('auth.register');
    }

    public function loginSubmit(Request $request)
    {
        $this->validate($request, [
            'email' => 'required',
            'password' => 'required',
        ]);

        $user = User::where('email', $request->email)
            ->first();
        if ($user == null) {
            return redirect()->back()->with('error', 'Email or phone not in stock');
        }

        if (Hash::check($request->password, $user->password)) {
            Auth::login($user);
            if ($user->type == 'admin') {
                return redirect()->intended('admin/dashboard');
            } else {
                return redirect()->intended('user/dashboard');
            }


        } else {
            return redirect()->back()->with('error', 'Password not match');
        }
    }

    public function logout()
    {
        Auth::logout();
        return redirect('login');
    }

    public function registerStore(Request $request)
    {

        $rules = [
            'name' => 'required|max:50',
            'email' => 'required|email|unique:users,email|max:255',
            'phone' => 'required|numeric|digits:11|regex:/^01[0-9]{9}$/|unique:users,phone',
            'password' => 'required|string|min:8|confirmed',
            'terms' => 'accepted',
        ];


        $messages = [
            'name.required' => 'দয়া করে আপনার নাম লিখুন।',
            'email.required' => 'ইমেইল প্রয়োজন।',
            'email.email' => 'একটি বৈধ ইমেইল ঠিকানা লিখুন।',
            'email.unique' => 'এই ইমেইলটি ইতিমধ্যে নিবন্ধিত।',
            'phone.required' => 'ফোন নম্বর প্রয়োজন।',
            'phone.numeric' => 'ফোন নম্বরটি সংখ্যাসূচক হতে হবে।',
            'phone.digits' => 'ফোন নম্বরটি অবশ্যই ১১ সংখ্যার হতে হবে।',
            'phone.unique' => 'এই ফোন নম্বরটি ইতিমধ্যে নিবন্ধিত।',
            'password.required' => 'পাসওয়ার্ড প্রয়োজন।',
            'password.min' => 'পাসওয়ার্ড কমপক্ষে ৮ অক্ষরের হতে হবে।',
            'password.confirmed' => 'পাসওয়ার্ড মেলেনি।',
            'terms.accepted' => 'শর্তাবলী মেনে নিতে হবে।',
        ];


        $validator = \Validator::make($request->all(), $rules, $messages);


        if ($validator->fails()) {
            foreach ($validator->errors()->all() as $error) {
                toastr()->error($error);
            }
            return redirect()->back()->withInput();
        }

        try {
            // Create a new user
            $newuser = new User();
            $newuser->name = $request->name;
            $newuser->type = 'user';
            $newuser->email = $request->email;
            $newuser->phone = $request->phone;
            $newuser->password = bcrypt($request->password);
            $newuser->save();
            Auth::login($newuser);
            toastr()->success('অ্যাকাউন্ট সফলভাবে তৈরি হয়েছে!');

            $nuser = User::where('type', 'admin')->first();
            Notification::send($nuser, new RegisterNotification($request));

            return redirect('user/dashboard');

        } catch (\Exception $e) {
            toastr()->error('অ্যাকাউন্ট তৈরি করতে সমস্যা হয়েছে। আবার চেষ্টা করুন!');
            return redirect()->back();
        }
    }


}
