<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Order;
use App\Models\AutoTopup;
use App\Models\Payment;
use App\Models\Voucher;
use App\Models\Vseed;
use Carbon\Carbon;
use Illuminate\Support\Str;

class Topup extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:topup';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'This command run for auto topup';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        Log::info(str_repeat('-', 50)); // 50 dashes
        Log::info('Starting Auto Topup');
        $this->checkStatus();
    
        // Process old orders
        $oldorders = Order::where('auto_status', 0)
                          ->whereNotNull('automatic')
                          ->where('created_at', '<=', Carbon::now()->subMinutes(5))
                          ->get();
    
        foreach ($oldorders as $oldorder) {
            $oldorder->auto_status = 3;
    
            // If the order has vouchers, restock them
            if ($oldorder->voucher_id) {
                $voucher = Voucher::find($oldorder->voucher_id);
                if ($voucher) {
                    $voucher->status = 0; // Restock the voucher
                    Log::info('Stock Added');
                    $voucher->save();
                }
            }
    
            $oldorder->save();
        }
    
        // Process new UnPaid orders
        $orders = Order::where('auto_status', 0)
                       ->whereNotNull('automatic')
                       ->take(5)
                       ->get();
    
        foreach ($orders as $order) {
            Log::info(str_repeat('-', 50)); // 50 dashes
            Log::info('Processing Order ID: ' . $order->id);
    
            if (isset($order)) {
                if ($order->payment_type == 'payment_gateway') {
                    $payment = Payment::where('order_id', $order->id)
                                      ->where('payment_status', 'pending')
                                      ->first();
                    if (isset($payment)) {
                        Log::info("Skipping Order ID {$order->id} due to pending payment.");
                        Log::info(str_repeat('-', 50)); // 50 dashes
                        continue; // Skip to the next order
                    }
                }
            }
    
            if (!$order) {
                Log::warning("Order not found or invalid.");
                continue;
            }
    
            Log::info("Order ID {$order->id} is being processed.");
    
            $orderId = $order->id;
            $playerId = $order->player_id;
            $voucher = Voucher::find($order->voucher_id);
    
            Log::info('Player ID: ' . $playerId);
            Log::info('Voucher Details: ' . json_encode($voucher));
    
            $vseed = Vseed::where('name', $voucher->automatic)->first();
    
            // Determine voucher type and string
            $type = Str::contains($voucher->code, 'BDMB-') ? '1' : '2';
            $string = Str::contains($voucher->code, 'BDMB-') ? str_replace('BDMB-', '', $voucher->code) : str_replace('UPBD-', '', $voucher->code);
    
            // Convert denomination to string
            $denom = $this->getDenominationString($vseed->denom);
    
            // Prepare the payload for the POST request
            $payload = [
                'playerId' => $playerId,
                'denom' => $denom,
                'type' => $type,
                'voucherCode' => $string,
                'webhook' => 'https://antsy-airport-01.webhook.cool',
            ];
    
            // Log the payload being sent for debugging
            Log::info('Sending top-up request', $payload);
    
            // Send the POST request for top-up
            $response = Http::withHeaders([
                'RA-SECRET-KEY' => 'bXktYlhrdGJtRnRaUzB4YlhrdGJtRnRaUzB4YlhrdGJtRnRaUzB4LTE=',
                'Content-Type' => 'application/json',
            ])->post('https://api.hidetopup.com/topup', $payload);
    
            // Log the response for debugging
            Log::info('Response received', ['response' => $response->json()]);
    
            // Check for response status
            if ($response->failed()) {
                $order->auto_status = '3';
                $order->save();
                Log::error("Failed to process top-up, Error: {$response->body()}");
                continue; // Handle the error appropriately
            }
    
            // Process the response data
            $responseData = $response->json();
            if (isset($responseData['status']) && $responseData['status'] === true) {
                Log::info('Top-up successful', $responseData);
    
                // Store the top-up details in the auto_topup table
                $this->storeTopupDetails($orderId, $responseData['uid'], 'pending');
                $order->auto_status = '1';
                $order->save();
    
                // Now check the transaction status using the UID
                $transactionUid = $responseData['uid'];
            } else {
                Log::error('Top-up failed', $responseData);
            }
        }
    }
    
    /**
     * Get the string representation of the denomination.
     *
     * @param int $denom
     * @return string|null
     */
    private function getDenominationString(int $denom): ?string
    {
        $denominationStrings = [
            0 => '0', 1 => '1', 2 => '2', 3 => '3', 4 => '4',
            5 => '5', 6 => '6', 7 => '7', 8 => '8', 9 => '9',
            10 => '10'
        ];
    
        return $denominationStrings[$denom] ?? null;
    }
    

    private function storeTopupDetails($orderId, $uid, $status): void
    {
        AutoTopup::create([
            'order_id' => $orderId,
            'uid' => $uid,
            'status' => $status,
        ]);
        // প্রতিটি অর্ডারের লগ শুরু করার আগে একটি ড্যাশ লাইনের ব্যবহারে স্পেস যোগ করা
        Log::info(str_repeat('-', 50)); // 50টি ড্যাশ
        Log::info("Stored top-up details: Order ID {$orderId}, UID {$uid}, Status {$status}");
    }



 


    public function checkStatus()
    {
        $autotopups = AutoTopup::where('status', 'pending')->get();
    
        foreach ($autotopups as $autotopup) {
            if (!$autotopup) {
                continue; // Skip if no AutoTopup record found
            }
    
            Log::info("Checking transaction status for UID: {$autotopup->uid}");
    
            $response = Http::withHeaders([
                'RA-SECRET-KEY' => 'bXktYlhrdGJtRnRaUzB4YlhrdGJtRnRaUzB4YlhrdGJtRnRaUzB4LTE=',
                'Content-Type' => 'application/json',
            ])->get("https://api.hidetopup.com/transactions/{$autotopup->uid}");
    
            // Log API response for debugging
            Log::info("API Response for UID {$autotopup->uid}: " . json_encode($response->json()));
    
            if ($response->failed()) {
                Log::error("API request failed for UID {$autotopup->uid}. Response: " . $response->body());
                continue; // Skip processing if API request failed
            }
    
            $transactionData = $response->json();
    
            // Ensure response has 'status'
            if (!isset($transactionData['status'])) {
                Log::error("Invalid response format for UID {$autotopup->uid}: " . json_encode($transactionData));
                continue;
            }
    
            $order = Order::find($autotopup->order_id);
            if (!$order) {
                Log::error("Order not found for AutoTopup ID {$autotopup->id}");
                continue;
            }
    
            if (strtolower($transactionData['status']) === 'failed') {
                Log::error("Transaction failed for UID {$autotopup->uid}: " . $transactionData['message']);
    
                // ✅ Update order status to "3" (Failed)
                $order->auto_status = '3';
                $order->save();
    
                // ✅ Update AutoTopup record
                $autotopup->status = 'failed';
                $autotopup->message = $transactionData['message'] ?? 'No message provided';
                $autotopup->save();
    
                Log::info("Order ID {$order->id} marked as failed (auto_status = 3).");
                continue; // Move to the next AutoTopup record
            }
    
            if (strtolower($transactionData['status']) === 'success') {
                Log::info("Transaction successful for UID {$autotopup->uid}");
    
                // ✅ Update order status to "2" (Success)
                $order->auto_status = '2';
                $order->save();
    
                // ✅ Update AutoTopup record
                $autotopup->status = 'success';
                $autotopup->save();
    
                Log::info("Order ID {$order->id} marked as success (auto_status = 2).");
            }
        }
    }
    



}







