<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Models\Order;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Setting;

class SendWhatsAppMessage implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $tries = 3;

    protected $order;

    public function __construct(Order $order)
    {
        $this->order = $order;
    }

    public function handle()
    {
        try {
            // Load only the user and variation relationships
            $order = $this->order->load(['user', 'variation']);
            
            if (!$order->exists) {
                Log::error("WhatsApp message failed: Order {$this->order->id} not found");
                return;
            }

            if (!$order->user || empty($order->user->phone)) {
                Log::error("WhatsApp message failed: User or phone not found for Order {$order->id}");
                return;
            }

// Fetch settings directly from the Setting model
$apiKey = Setting::where('key', 'whatsapp_message_api')->value('value') ?? 'null';
$sender = Setting::where('key', 'whatsapp_sender_number')->value('value') ?? 'null';
$adminNumber = Setting::where('key', 'whatsapp_admin_number')->value('value') ?? 'null';


            $userPhone = $this->formatPhoneNumber($order->user->phone);
            $message = $this->composeOrderMessage($order);

            $this->sendWhatsAppMessage($apiKey, $sender, $userPhone, $message, $order->id, 'user');

            if ((string)$adminNumber !== $userPhone) {
                $adminMessage = "New Order Alert!\n\n" . $message;
                $this->sendWhatsAppMessage($apiKey, $sender, $adminNumber, $adminMessage, $order->id, 'admin');
            }

        } catch (\Exception $e) {
            Log::error("Exception in SendWhatsAppMessage job", [
                'order_id' => $this->order->id ?? 'unknown',
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    private function formatPhoneNumber($phoneNumber, $defaultCountryCode = '880')
    {
        $phone = preg_replace('/\s+/', '', ltrim($phoneNumber, '+'));
        if (!preg_match('/^\d{12,13}$/', $phone)) {
            $phone = $defaultCountryCode . ltrim($phone, '0');
        }
        return $phone;
    }

    private function composeOrderMessage(Order $order)
    {
        $variationTitle = $order->variation ? ($order->variation->title ?? '[No Variation Title]') : '[No Variation]';

        return "Order Confirmation\n\n" .
               "Order ID: " . ($order->id ?? 'N/A') . "\n" .
               "Product Title : - {$variationTitle}\n" .
               "Type: " . ($order->type ?? 'N/A') . "\n" .
               "Player ID: " . ($order->player_id ?? 'Other') . "\n" .
               "Amount: " . ($order->total_price ?? 0) . " BDT\n" .
               "Thank you for your purchase!";
    }

    private function sendWhatsAppMessage($apiKey, $sender, $recipient, $message, $orderId, $recipientType)
    {
        $response = Http::post('https://wa.smmclouds.com/send-message', [
            'api_key' => $apiKey,
            'sender' => $sender,
            'number' => $recipient,
            'message' => $message,
        ]);

        if ($response->failed()) {
            Log::error("WhatsApp message failed for Order {$orderId} ({$recipientType})", [
                'status' => $response->status(),
                'response' => $response->body(),
                'recipient' => $recipient
            ]);
            throw new \Exception("Failed to send WhatsApp message to {$recipient}");
        }

        Log::info("WhatsApp message sent successfully for Order {$orderId} ({$recipientType})", [
            'recipient' => $recipient
        ]);
    }
}