<?php

namespace App\Http\Controllers\admin;

use App\Models\User;
use App\Models\Payment;
use App\Models\Setting;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Models\PaymentGateway;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class AdminController extends Controller
{
   

  



    public function usersList()
    {
        // Fetching all users
        $users = User::paginate(25);
        
       
        return view('admin.users_list', compact('users'));
    }


    public function updateBalance(Request $request)
    {
        \Log::info('Request Data: ' . json_encode($request->all()));

        $request->validate([
            'user_id' => 'required|exists:users,id',
            'amount' => 'required|numeric|min:0',
        ]);

        $user = User::findOrFail($request->user_id);
        \Log::info('Resolved User ID: ' . $user->id);

        $previous_balance = $user->balance;
        $user->balance += $request->amount;
        $user->save();

        $new_balance = $user->balance;
        $amount = $request->amount;

        Transaction::create([
            'user_id' => $user->id,
            'previous_balance' => $previous_balance,
            'amount' => $amount,
            'new_balance' => $new_balance,
            'transaction_id' => uniqid('admin_added_'),
            'payment_status' => 'completed',
        ]);

        return response()->json(['success' => 'Balance updated successfully.']);
    } // End Method


    public function userWalletHistory($id)
    {
        $user = User::findOrFail($id);
    
       // Fetch all completed transactions, sorted by date (oldest first)
       $transactions = Transaction::where('user_id', $id)
       ->where('payment_status', 'completed')
       ->orderBy('created_at', 'asc')
       ->get();
        return view('admin.transaction.wallet-history', compact('user', 'transactions'));
    }
     // End Method
    


// Update user details
public function updateUserDetails(Request $request)
{
    $request->validate([
        'user_id' => 'required|exists:users,id',
        'name' => 'nullable|string|max:255',
        'email' => 'nullable|email|max:255',
        'phone' => 'nullable|string|max:15',
        'password' => 'nullable|string|min:6', // Allow empty password field
    ]);

    $user = User::find($request->user_id);
    $user->name = $request->name;
    $user->email = $request->email;
    $user->phone = $request->phone;

    if ($request->filled('password')) {
        $user->password = Hash::make($request->password); // Update password if provided
    }

    $user->save();

    return response()->json(['success' => 'User details updated successfully.']);
} // End Method


// Reset user password
public function resetPassword(Request $request)
{
    $request->validate([
        'user_id' => 'required|exists:users,id',
        'password' => 'required|min:6',
    ]);

    $user = User::find($request->user_id);
    $user->password = Hash::make($request->password);
    $user->save();

    return response()->json(['success' => 'Password reset successfully.']);
} // End Method



    public function index()
    {
        return view('admin.index');
    } // End Method

    public function AdminLogout(Request $request)
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        $notification = [
            'message'    => 'Admin Logged Out Successfully',
            'alert-type' => 'success',
        ];

        return redirect('/admin/login')->with($notification);
    } // End Method

    public function AdminLogin()
    {

        return view('admin.login.admin_login');
    } // End Method





    public function AdminProfile()
    {

        $id          = Auth::user()->id;
        $profileData = User::find($id);

        return view('admin.admin_profile_view', compact('profileData'));
    } // End Method

    public function AdminProfileStore(Request $request)
    {

        $id             = Auth::user()->id;
        $data           = User::find($id);
        $data->name     = $request->name;
        $data->email    = $request->email;
        $data->phone    = $request->phone;
      
        if ($request->file('photo')) {
            $file        = $request->file('photo');
            @unlink(public_path('upload/admin_images/' . $data->photo));
            $filename    = date('YmdHi') . $file->getClientOriginalName();
            $file->move(public_path('upload/admin_images'), $filename);

            $data->photo = $filename;
        }

        $data->save();

        $notification   = [
            'message'    => 'Admin Profile Updated Successfully',
            'alert-type' => 'success',
        ];

        return redirect()->back()->with($notification);
    } // End Method

    public function AdminChangePassword()
    {

        $id          = Auth::user()->id;
        $profileData = User::find($id);

        return view('admin.admin_change_password', compact('profileData'));
    } // End Method

    public function AdminUpdatePassword(Request $request)
    {

        // Validation
        $request->validate([
            'old_password' => 'required',
            'new_password' => 'required|confirmed',

        ]);

        /// Match The Old Password

        if (!Hash::check($request->old_password, auth::user()->password)) {

            $notification = [
                'message'    => 'Old Password Does not Match!',
                'alert-type' => 'error',
            ];

            return back()->with($notification);
        }

        /// Update The New Password

        User::whereId(auth()->user()->id)->update([
            'password' => Hash::make($request->new_password),

        ]);

        $notification = [
            'message'    => 'Password Change Successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    } // End Method

    public function AllUser()
    {
        $users = User::all();

        return view('backend.user.all_user', compact('users'));
    } // End Method

 

    public function showPaymentGateways()
    {
        // Fetch all payment gateways from the database
        $paymentGateways = PaymentGateway::all();
    
        // Pass the payment gateways to the view
        return view('admin.payment.gateways_index', compact('paymentGateways'));
    }



    public function updatePaymentGateways(Request $request)
    {
        // Validate inputs
        $request->validate([
            'api_key' => 'array',
            'client' => 'array',
            'secret' => 'array',
            'base_url' => 'array',
            'active_gateway' => 'nullable|exists:payment_gateways,id',
        ]);
    
        // Loop through each submitted gateway
        foreach ($request->api_key as $gatewayId => $apiKey) {
            PaymentGateway::where('id', $gatewayId)->update([
                'api_key' => $apiKey,
                'client' => $request->client[$gatewayId] ?? null,
                'secret' => $request->secret[$gatewayId] ?? null,
                'base_url' => $request->base_url[$gatewayId] ?? null,
            ]);
        }
    
        // Reset all gateways to inactive
        PaymentGateway::query()->update(['is_active' => 0]);
    
        // Activate the selected gateway
        if ($request->active_gateway) {
            PaymentGateway::where('id', $request->active_gateway)->update(['is_active' => 1]);
        }
    
        return redirect()->back()->with('success', 'Payment gateways updated successfully!');
    }


    

    public function showBkash()
{
    // Fetch Bkash settings directly from the database
    $bkashSettings = Setting::whereIn('key', [
        'bkash_username',
        'bkash_password',
        'bkash_app_key',
        'bkash_app_secret_key',
        'bkash_merchant_enable' // Include enable/disable status
    ])->pluck('value', 'key');

    return view('admin.payment.bkash_gateways_index', compact('bkashSettings'));
} // End Method


public function BkashGetwayUpdate(Request $request)
{
    // Validate Input Fields 
    $request->validate([
        'bkash_username' => 'required|string|max:255',
        'bkash_password' => 'required|string|max:255',
        'bkash_app_key' => 'required|string|max:255',
        'bkash_app_secret_key' => 'required|string|max:255',
        'bkash_merchant_enable' => 'required|in:0,1' // Validate enable/disable selection
    ]);

    // Save settings directly (no caching)
    $settings = [
        'bkash_username' => $request->bkash_username,
        'bkash_password' => $request->bkash_password,
        'bkash_app_key' => $request->bkash_app_key,
        'bkash_app_secret_key' => $request->bkash_app_secret_key,
        'bkash_merchant_enable' => $request->bkash_merchant_enable, // Enable/Disable status
    ];

    foreach ($settings as $key => $value) {
        Setting::updateOrCreate(['key' => $key], ['value' => $value]);
    }

    // ❌ No cache clearing needed

    return redirect()->back()->with('warning', 'Bkash settings updated successfully!');
}  // End Method
    



 public function blockUser($id)
 {
     $user = User::findOrFail($id);
     $user->blocked = true;
     $user->save();
 
     return redirect()->back()->with('success', 'User has been blocked.');
 }
 
 public function unblockUser($id)
 {
     $user = User::findOrFail($id);
     $user->blocked = false;
     $user->save();
 
     return redirect()->back()->with('success', 'User has been unblocked.');
 }


}