<?php

namespace App\Console\Commands;

use Carbon\Carbon;
use App\Models\Order;
use App\Models\Payment;
use App\Models\Voucher;
use App\Models\Variation;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CancelledOrderRestockCommand extends Command
{
    protected $signature = 'app:restock-cancelled-orders';
    protected $description = 'Remove unpaid orders, restock vouchers, and delete payments after 3 minutes';

    public function handle()
    {
        Log::info("Starting order removal process");

        $thresholdTime = Carbon::now()->subMinutes(3);
        
        $pendingOrders = Order::with('payment')
            ->where('status', 0)
            ->whereNotNull('voucher_id')
            ->whereNull('automatic')
            ->where('created_at', '<=', $thresholdTime)
            ->get();

        if ($pendingOrders->isEmpty()) {
            Log::info("No orders to remove");
            return;
        }

        foreach ($pendingOrders as $order) {
            $this->processOrderRemoval($order);
        }

        Log::info("Order removal process completed");
    }

    protected function processOrderRemoval(Order $order)
    {
        DB::beginTransaction();
        try {
            // 1. Restock voucher and variation
            if ($voucher = Voucher::find($order->voucher_id)) {
                $voucher->update(['status' => 0]);
                
                if ($variation = Variation::find($voucher->variation_id)) {
                    $variation->increment('stock', $order->quantity);
                    Log::info("Restocked variation {$variation->id}");
                }
            }

            // 2. Delete payment record
            if ($order->payment) {
                $order->payment->delete();
                Log::info("Deleted payment for order #{$order->id}");
            }

            // 3. Delete the order itself
            $order->delete();
            Log::info("Permanently removed order #{$order->id}");

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Failed to remove order #{$order->id}: " . $e->getMessage());
        }
    }
}