<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class OrdersTableSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $users = DB::table('users')->pluck('id'); // Get all user IDs
        $vouchers = DB::table('vouchers')
            ->where('status', 0) // Fetch only available vouchers
            ->get();

        if ($vouchers->isEmpty()) {
            $this->command->info('No available vouchers for orders.');
            return;
        }

        $orders = [];
        foreach ($users as $userId) {
            if ($vouchers->isEmpty()) {
                $this->command->info('No more vouchers available for assignment.');
                break;
            }

            // Get a random available voucher
            $voucher = $vouchers->random();

            // Prepare the order
            $orders[] = [
                'user_id' => $userId,
                'order_sumid' => $this->generateUniqueOrderSumId(),
                'invoice_id' => null,
                'product_id' => $voucher->product_id,
                'variation_id' => $voucher->variation,
                'voucher_id' => $voucher->id,
                'single_price' => $voucher->price,
                'total_price' => $voucher->price,
                'type' => 'voucher',
                'qty' => 1,
                'status' => 2, // Completed
                'auto_status' => 0,
                'notice' => null,
                'payment_type' => 'wallet',
                'player_id' => null,
                'automatic' => $voucher->automatic,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ];

            // Update the voucher status to "used" (1)
            DB::table('vouchers')
                ->where('id', $voucher->id)
                ->update(['status' => 1]);

            // Remove the assigned voucher from the collection
            $vouchers = $vouchers->reject(fn($v) => $v->id === $voucher->id);
        }

        // Bulk insert orders
        DB::table('orders')->insert($orders);

        $this->command->info(count($orders) . ' orders have been created.');
    }

    private function generateUniqueOrderSumId()
    {
        do {
            $orderId = rand(1000000000, 9999999999);
        } while (DB::table('orders')->where('order_sumid', $orderId)->exists());

        return $orderId;
    }
    }

