<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Models\Order;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Setting;

class SendWhatsAppMessage implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $tries = 3;

    protected $order;

    public function __construct(Order $order)
    {
        $this->order = $order;
    }

    public function handle()
    {
        try {
            $order = $this->order->load(['user', 'variation']);
            
            if (!$order->exists) {
                Log::error("WhatsApp message failed: Order {$this->order->id} not found");
                return;
            }

            $apiKey = Setting::where('key', 'whatsapp_message_api')->value('value') ?? 'null';
            $sender = Setting::where('key', 'whatsapp_sender_number')->value('value') ?? 'null';
            $adminNumber = Setting::where('key', 'whatsapp_admin_number')->value('value') ?? 'null';

            $userPhone = null;
            if ($order->user && !empty($order->user->phone)) {
                $userPhone = $this->formatPhoneNumber($order->user->phone);
            } else {
                Log::error("WhatsApp message failed: User or phone not found for Order {$order->id}");
            }

            $message = $this->composeOrderMessage($order);

            // Send to user if phone exists
            if ($userPhone && $this->isWhatsAppNumber($apiKey, $sender, $userPhone)) {
                $this->sendWhatsAppMessage($apiKey, $sender, $userPhone, $message, $order->id, 'user');
            } elseif (!$userPhone) {
                Log::info("Skipped user message for Order {$order->id} - No valid user phone");
            } else {
                Log::info("Skipped user message for Order {$order->id} - Number not registered on WhatsApp", [
                    'recipient' => $userPhone
                ]);
            }

            // Send to admin regardless of user phone
            if ($adminNumber && (string)$adminNumber !== $userPhone) {
                $adminMessage = "New Order Alert!\n\n" . 
                                ($userPhone ? "" : "WARNING: User phone not found!\n\n") . 
                                $message;
                if ($this->isWhatsAppNumber($apiKey, $sender, $adminNumber)) {
                    $this->sendWhatsAppMessage($apiKey, $sender, $adminNumber, $adminMessage, $order->id, 'admin');
                } else {
                    Log::info("Skipped admin message for Order {$order->id} - Admin number not registered on WhatsApp", [
                        'recipient' => $adminNumber
                    ]);
                }
            }

        } catch (\Exception $e) {
            Log::error("Exception in SendWhatsAppMessage job", [
                'order_id' => $this->order->id ?? 'unknown',
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    private function formatPhoneNumber($phoneNumber, $defaultCountryCode = '880')
    {
        $phone = preg_replace('/\s+/', '', ltrim($phoneNumber, '+'));
        if (!preg_match('/^\d{12,13}$/', $phone)) {
            $phone = $defaultCountryCode . ltrim($phone, '0');
        }
        return $phone;
    }

    private function composeOrderMessage(Order $order)
    {
        $variationTitle = $order->variation ? ($order->variation->title ?? '[No Variation Title]') : '[No Variation]';

        return "Order Confirmation\n\n" .
               "Order ID: " . ($order->id ?? 'N/A') . "\n" .
               "Product Title : - {$variationTitle}\n" .
               "Type: " . ($order->type ?? 'N/A') . "\n" .
               "Player ID: " . ($order->player_id ?? 'Other') . "\n" .
               "Amount: " . ($order->total_price ?? 0) . " BDT\n" .
               "Thank you for your purchase!";
    }

    private function isWhatsAppNumber($apiKey, $sender, $number)
    {
        // Placeholder for actual WhatsApp check endpoint
        $response = Http::post('https://wa.smmclouds.com/check-number', [
            'api_key' => $apiKey,
            'sender' => $sender,
            'number' => $number,
        ]);

        if ($response->successful()) {
            $data = $response->json();
            return $data['status'] === true; // Adjust based on actual API response
        }
        return true; // Fallback: assume true and handle in sendWhatsAppMessage
    }

    private function sendWhatsAppMessage($apiKey, $sender, $recipient, $message, $orderId, $recipientType)
    {
        $response = Http::post('https://wa.smmclouds.com/send-message', [
            'api_key' => $apiKey,
            'sender' => $sender,
            'number' => $recipient,
            'message' => $message,
        ]);

        if ($response->failed()) {
            $errorData = $response->json();
            $errorMessage = $errorData['msg'] ?? 'Unknown error';

            if (str_contains(strtolower($errorMessage), 'not registered') || $response->status() === 400) {
                Log::info("WhatsApp message skipped for Order {$orderId} ({$recipientType}) - Number not on WhatsApp", [
                    'recipient' => $recipient,
                    'response' => $response->body()
                ]);
                return;
            }

            Log::error("WhatsApp message failed for Order {$orderId} ({$recipientType})", [
                'status' => $response->status(),
                'response' => $response->body(),
                'recipient' => $recipient
            ]);
            throw new \Exception("Failed to send WhatsApp message to {$recipient}");
        }

        Log::info("WhatsApp message sent successfully for Order {$orderId} ({$recipientType})", [
            'recipient' => $recipient
        ]);
    }
}