<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Artisan;
use ZipArchive;

class UpdateController extends Controller
{
    public function update()
    {
        try {
            // Fetch version.json from server
            $response = Http::get('https://pay.rexgamingbd.com/version.json');

            if ($response->failed()) {
                throw new \Exception("Failed to fetch version.json. Server error or file not found.");
            }

            $remoteVersion = json_decode($response->body(), true);
            if (!isset($remoteVersion['version']) || !isset($remoteVersion['update_url'])) {
                throw new \Exception("Invalid version.json format.");
            }

            // Get current version from .env
            $currentVersion = config('app.version');

            // Check if update is required
            if ($remoteVersion['version'] > $currentVersion) {
                // Download update file
                if (!$this->downloadUpdate($remoteVersion['update_url'])) {
                    throw new \Exception("Failed to download update file.");
                }

                // Extract and apply update
                if (!$this->applyUpdate()) {
                    throw new \Exception("Failed to extract and apply update.");
                }

                // Run migrations
                 Artisan::call('migrate', ['--force' => true]);

                // Clear Laravel caches
                Artisan::call('config:clear');
                Artisan::call('cache:clear');
                Artisan::call('view:clear');
                Artisan::call('route:clear');
                Artisan::call('optimize:clear');

                // Update the .env file with new version
                $this->updateEnvVersion($currentVersion, $remoteVersion['version']);

                return back()->with('success', 'System updated successfully to version ' . $remoteVersion['version']);
            } else {
                return back()->with('info', 'You are already using the latest version.');
            }

        } catch (\Exception $e) {
            return back()->with('error', 'Update failed: ' . $e->getMessage());
        }
    }

    private function downloadUpdate($updateUrl)
    {
        $zipPath = storage_path('app/update.zip');

        // Check if the file exists before downloading
        if (!@file_get_contents($updateUrl, false, stream_context_create(["http" => ["ignore_errors" => true]]))) {
            return false;
        }

        return file_put_contents($zipPath, file_get_contents($updateUrl)) !== false;
    }

    private function applyUpdate()
    {
        $zipPath = storage_path('app/update.zip');
        $extractPath = storage_path('app/update_extract/');

        $zip = new ZipArchive;
        if ($zip->open($zipPath) === TRUE) {
            // Ensure the extraction path exists
            if (!file_exists($extractPath)) {
                mkdir($extractPath, 0777, true);
            }

            $zip->extractTo($extractPath);
            $zip->close();
            unlink($zipPath); // Delete ZIP after extraction

            // Move only contents of "update/" to Laravel root
            $updateFolder = $extractPath . "/update";
            if (file_exists($updateFolder)) {
                $this->recursiveCopy($updateFolder, base_path());
                exec("rm -rf " . escapeshellarg($extractPath)); // Cleanup extracted files
            } else {
                throw new \Exception("The extracted update does not contain the expected 'update/' directory.");
            }

            return true;
        } else {
            return false;
        }
    }

    // Function to recursively copy files and overwrite existing ones
    private function recursiveCopy($src, $dst)
    {
        $dir = opendir($src);
        @mkdir($dst, 0777, true);

        while (false !== ($file = readdir($dir))) {
            if ($file !== '.' && $file !== '..') {
                if (is_dir($src . '/' . $file)) {
                    $this->recursiveCopy($src . '/' . $file, $dst . '/' . $file);
                } else {
                    copy($src . '/' . $file, $dst . '/' . $file);
                }
            }
        }
        closedir($dir);
    }

    private function updateEnvVersion($oldVersion, $newVersion)
    {
        $envPath = base_path('.env');
        if (file_exists($envPath)) {
            file_put_contents($envPath, str_replace(
                'APP_VERSION=' . $oldVersion,
                'APP_VERSION=' . $newVersion,
                file_get_contents($envPath)
            ));
        }
    }
}
